<?php 
namespace addie\storage;

//Initiate the Google FireStore Class 
use Google\Cloud\Storage\StorageClient;

#

/*
            _     _ _   __         _                          __         _                 _     _                             
           | |   | (_)  \ \       | |                         \ \       | |               | |   | |                            
   __ _  __| | __| |_  __\ \   ___| |_ ___  _ __ __ _  __ _  __\ \   ___| | ___  _   _  __| |___| |_ ___  _ __ __ _  __ _  ___ 
  / _` |/ _` |/ _` | |/ _ \ \ / __| __/ _ \| '__/ _` |/ _` |/ _ \ \ / __| |/ _ \| | | |/ _` / __| __/ _ \| '__/ _` |/ _` |/ _ \
 | (_| | (_| | (_| | |  __/\ \\__ \ || (_) | | | (_| | (_| |  __/\ \ (__| | (_) | |_| | (_| \__ \ || (_) | | | (_| | (_| |  __/
  \__,_|\__,_|\__,_|_|\___| \_\___/\__\___/|_|  \__,_|\__, |\___| \_\___|_|\___/ \__,_|\__,_|___/\__\___/|_|  \__,_|\__, |\___|
                                                       __/ |                                                         __/ |     
                                                      |___/                                                         |___/                                                  

What? The http\storage\cloudstorage class is built to help with managing Google Cloud Storage buckets. GCP Cloud storage offers great pricing for storage.
Who? Developed by Don Hawkins
*/

class cloudstorage {

public $bucket;
public $app_credentials_path;



public function __construct($bucket,$app_credentials_path) {
        
  $this->bucket = trim($bucket);
  if ($content = file_get_contents($app_credentials_path)) { //Google Application credentials, point to the file (doesn't have to be in this folder)

    //If the file is remote we need to download it to a local file.
    if (str_contains($app_credentials_path,'http://') || str_contains($app_credentials_path,'https://')) {
      @mkdir('/tmp');
      if (!file_exists('/tmp/google_cloud_storage_credentials.json')) { file_put_contents('/tmp/google_cloud_storage_credentials.json',$content); };
      $app_credentials_path = '/tmp/google_cloud_storage_credentials.json';
    }

    putenv("GOOGLE_APPLICATION_CREDENTIALS=$app_credentials_path");
  }else{
    throw new \Exception('Unable to locate app credentials at the specified url.'); 
  }
  $this->storage = new StorageClient();

}

    //Check if a file exists in the specified bucket.
    public function exists($filename) {

      $bucket = $this->storage->bucket($this->bucket);
      $object = $bucket->object($filename);

      if ($object->exists()) { return true; }else{ return false; }

    }

    # Get a file. Specify $method as 'file' to download the file to the specified $destination_file or as 'contents' to store the contents of the file in a sting and return the value.
    public function get($filename,$destination_file,$method = 'file') {

      if (strtolower($method) == 'file') {
        $bucket = $this->storage->bucket($this->bucket);
        $object = $bucket->object($filename);
        $object->downloadToFile($destination_file);
        if (exist($destination_file)) { return true; }else{ throw new \Exception('File not present in destination.'); }
      }
      if (strtolower($method) == 'contents') {
        $bucket = $this->storage->bucket($this->bucket);
        $object = $bucket->object($filename);
        $contents = $object->downloadAsString($destination_file);
        if (!empty($contents)) { return $contents; }else{ throw new \Exception('Returns empty string.'); }
      }
      
    }

    ## Create a file with the specified 'filename' and upload the specified 'contents' to the file. Don't want the file overwritten if it exist? Specify $overwrite = false
    public function upload(string $filename,string $tmp_file_path,$overwrite = true) {

              //Check if the file exist.
              if ($overwrite === false) {
                if ($this->exists($filename)) { throw new \Exception("A file with the specified filename ($filename) already exist and overwrite is set to false."); }
              }

      if (file_exists($tmp_file_path)) { 

        $file = fopen($tmp_file_path, 'r');
        $bucket = $this->storage->bucket($this->bucket);
        $object = $bucket->upload($file, ['name' => $filename]);

        $file_url = 'https://storage.googleapis.com/' . $this->bucket . '/' . $filename;
        return $file_url;

      }else{

      throw new \Exception('Unable to locate file with path : ' . $tmp_file_path);

      };

    }

    ## Update an existing file by writing over it or adding content to the start or end of it.
    public function update(string $filename,string $contents) {

      if ($filename == '') { throw new \Exception("The 'update' function requires two parameters. Param 1 (filename) is missing."); }
      if (trim($contents) == '') { throw new \Exception("The 'update' function requires two parameters. Param 2 (contents) is missing."); }
      if (strlen($contents) < 3) { throw new \Exception("The 'update' function requires two parameters. Param 2 (contents) must be three or more characters."); }

      //Check if the file exist.
      if (!$this->exists($filename)) { throw new \Exception("A file with the specified filename ($filename) doesn't exist."); }

        $stream = fopen('data://text/plain,' . $contents, 'r');
        $bucket = $this->storage->bucket($this->bucket);
        $bucket->upload($stream, [
            'name' => $filename,
        ]);

        return true;
    
    }


    ## Renable a bucket object/file
    public function rename($filename,$new_filename) {

      if ($filename == '') { throw new \Exception("The 'rename' function requires two parameters. Param 1 (filename) to rename is missing."); }
      if ($new_filename == '') { throw new \Exception("The 'rename' function requires two parameters. Param 2 (new_filename) is missing."); }
      $bucket = $this->storage->bucket($this->bucket);
      $object = $bucket->object($filename);
      $object->copy($this->bucket, ['name' => $new_filename]);
      $object->delete();

      return true;
    
    }


    public function delete(string $filename) {
        $bucket = $this->storage->bucket($this->bucket);
        $object = $bucket->object($filename);
        $object->delete();
        return true;
    }



     ## Add a label to the bucket
     function addBucketLabel($label_name,$label_value) {

        if ($label_name == '') { throw new \Exception("The 'add_bucket_label' function requires two parameters. Param 1 (label name) is missing"); }
        if ($label_value == '') { throw new \Exception("The 'add_bucket_label' function requires two parameters. Param 2 (label value) is missing"); }
        $bucket = $this->storage->bucket($this->bucket);
        $newLabels = [$label_name => $label_value];
        $bucket->update(['labels' => $newLabels]);
        return true;

     }


     ## List all objects/files in the bucket and return as an array.
     public function listObjects() {

      $object_list = array();

         $bucket = $this->storage->bucket($this->bucket);
         foreach ($bucket->objects() as $object) {
          $object_list[] = $object->name();
         }

      return $object_list;

     }

    ## Make a file in a bucket publicly accessable.
    public function makePublic($filename) {

      if ($filename == '') { throw new \Exception("The 'makePublic' function requires param 1 (filename)."); }
      $bucket = $this->storage->bucket($this->bucket);
      $object = $bucket->object($filename);
      $object->update(['acl' => []], ['predefinedAcl' => 'PUBLICREAD']);
      $file_url = 'https://storage.googleapis.com/' . $this->bucket . '/' . $filename;
      return $file_url;
    }

    ## Create a new bucket
    public function createBucket(string $bucketName) {    
        $bucket = $this->storage->createBucket($this->bucket);
        return true;
    }

}
?>