<?php
namespace addie\http;

/*
            _     _ _   __     _     _   _     __                                    _   
           | |   | (_)  \ \   | |   | | | |    \ \                                  | |  
   __ _  __| | __| |_  __\ \  | |__ | |_| |_ _ _\ \   _ __ ___  __ _ _   _  ___  ___| |_ 
  / _` |/ _` |/ _` | |/ _ \ \ | '_ \| __| __| '_ \ \ | '__/ _ \/ _` | | | |/ _ \/ __| __|
 | (_| | (_| | (_| | |  __/\ \| | | | |_| |_| |_) \ \| | |  __/ (_| | |_| |  __/\__ \ |_ 
  \__,_|\__,_|\__,_|_|\___| \_\_| |_|\__|\__| .__/ \_\_|  \___|\__, |\__,_|\___||___/\__|
                                            | |                   | |                    
                                            |_|                   |_|                    

What? The http\request class is a curl wrapper used to make request to external url's. The class supports get, post, patch, and delete request. 
Who? Developed by Don Hawkins with inspiration from 'Guzzle' by @guzzle and 'Request' by @geerlingguy
*/

class request {

public $url;
public $timeout;
public $ssl = false;
public $userAgent = 'Mozilla/5.0 (compatible; Addie http library)';
public $max_redirects = 2;
public $debug;
public $contentType;
public $cookiePath;
public $rawData;
public $headers;
public $mimeType;
public $fileSize;
private $params;
private $credentials;
public $latency = 0;
public $proxyUrl;
public $proxyUsername;
public $proxyPassword;
private $httpCode;
private $responseBody;
private $responseHeader;
private $error;
private $decoded;
private $raw;


    public function __construct($url) {
      $this->url = $url;
    }


    ## request / send the request via Curl, requires the http_request method.
    public function request($method) {

        $method = strtoupper($method);
        
        $this->check_url();
      
        if (!in_array($method, ['GET', 'POST', 'PATCH', 'PUT', 'DELETE'])) { $method = 'GET'; }; // Default method is GET

        $curl = $this->http_request($method);
        return $curl;

    }

/*

    Set Functions

*/

    ## setCookiePath / Set the path to a text file where cookie information can be stored.
    public function setCookiePath($cookiePath) {

      $this->cookiePath = $cookiePath;
      return true;
    
    }


    ## setDebug / Set the to true or false to set 'CURLOPT_VERBOSE'.
    public function setDebug($debug) {

      if ($debug === true || $debug === false) { }else{ throw new \Exception('addie\http\request\setDebug : setDebug must be true or false.'); }

      $this->debug = $debug;
      return true;

    }


    ## setParams / Set an array of parameters. If using GET, fields will be appended to the end of the url (see private function 'build_query_string'). Unsets 'setBlock' if set.
    public function setParams($params = array()) {

        if (!is_array($params)) { throw new \Exception('addie\http\request\setParams : params must be an array.');  };
        
        //Check all array values and make sure a key is present (value can be empty)
        foreach ($params as $key => $this_param) {
          if (trim($key) == '') { throw new \Exception('addie\http\request\setParams : each parameter must have a key.'); };
        }

        if (isset($this->rawData)) { unset($this->rawData); };
        $this->params = $params;

        return true;
    }


    ## setRawData / Sometimes we want to POST a block of XML or JSON instead of parameters, this will set the 'block' of code and unset parameters.
    public function setRawData($rawData) {

        if (trim($rawData) == '') { throw new \Exception('addie\http\request\setRawData : rawData must be provided.');  };
  
      if (isset($this->params)) { unset($this->params); };
      $this->rawData = $rawData;

      return true;
    }

    ## setHeader / Set headers if needed.
    public function setHeader($headers = array()) {

      if (!is_array($headers)) { throw new \Exception('addie\http\request\setHeader : headers must be an array.');  };
    
      $this->headers = $headers;
      return true;

    }


    ## setCreds / For basic http authentication use this function to set credentials 
    public function setAuthCreds($username,$password) {

      if ($username == '' || $password == '') { throw new \Exception('addie\http\request\setAuthCreds : A username and password is required for basic http authentication.');  };
      
      $this->credentials = trim($username) . ':' . trim($password);
      return true;
    }


    ## setTimeout 
    public function setTimout($timeout) {

      if (!ctype_digit($timeout)) { throw new \Exception('addie\http\request\setTimeout : timeout value must be a numeric amount of seconds.'); };
      if ($timeout < 5) { throw new \Exception('addie\http\request\setTimeout : timeout must be 5 or above since the connect timout is already 5.'); }

      $this->timeout = $timeout;
      return true;

    }

    ## setSsl / Set the to true or false to verify the peer's ssl capability.
    public function setSsl($ssl) {

      if ($ssl === true || $ssl === false) { }else{ throw new \Exception('addie\http\request\setSsl : setSsl must be true or false.'); }

      $this->ssl = $ssl;
      return true;

    }


    ## maxRedirects / Maximum number of redirect allowed, follows are allowed by default.
    public function maxRedirects($max_redirects) {

      if (!ctype_digit($max_redirects)) { throw new \Exception('addie\http\request\maxRedirects : max redirect must be a numberic value between 0 and 10.'); };
      if ($max_redirects < 0 || $max_redirects > 10) { throw new \Exception('addie\http\request\maxRedirects: max redirect must be a numberic value between 0 and 10.'); }

      $this->max_redirects = $max_redirects;
      return true;

    }

    ## setUserAgent / Set the user agent if needed.
    public function setUserAgent($userAgent) {

      if ($userAgent == '') { throw new \Exception('addie\http\request\setUserAgent : setSUserAgent must not be empty.'); }

      $this->userAgent = $userAgent;

      return true;

    }

    ## setProxy / If using a proxy this function allows you to set the proxy url (ie. https://myproxyserver:8080)
    public function setProxy($proxyUrl) {

      if ($proxyUrl == '') { throw new \Exception('addie\http\request\setProxy : proxyUrl must not be empty.'); }

      $this->proxyUrl = $proxyUrl;

      return true;

    }

    ## setProxyAuth / If using a proxy that requires authentication.
    public function setProxyAuth($proxyUsername,$proxyPassword) {

      if ($this->proxyUrl == '') { throw new \Exception("addie\http\request\setProxyAuth : proxyUrl must be set first by calling the 'setProxy' function."); }
      if ($proxyUsername == '') { throw new \Exception('addie\http\request\setProxyAuth : proxyUsername must not be empty.'); }
      if ($proxyPassword == '') { throw new \Exception('addie\http\request\setProxyAuth : proxyPassword must not be empty.'); }

      $this->proxyUsername = $proxyUsername;
      $this->proxyPassword = $proxyPassword;

      return true;

    }

    ## responseContains / Check if the response (body) contains a specific string.
    public function responseContains($content = '') {
      if ($this->httpCode == 200 && !empty($this->responseBody)) {
        if (str_contains($this->responseBody, $content) !== FALSE) {
          return TRUE;
        }
      }
      return FALSE;
    }


    ## decode / Decode a xml or json string and convert it to a more friendly PHP object.
    public function decode($contentType) {
      
      $xml_errors = ''; $xml_errors_pre = '';

      if (!empty($this->responseBody)) { }else{ throw new \Exception('addie\http\request\decode : no body was detected in the response.'); }

      //Convert XML to object
      if ($contentType == 'xml') {

          libxml_use_internal_errors(true);
          $decoded = simplexml_load_string($this->responseBody);

          if ($decoded === false) {
            foreach(libxml_get_errors() as $error) {
                $xml_errors_pre = "\t" . $error->message;
                $xml_errors = $xml_errors_pre . ' ' . $xml_errors;
            }

              if (is_object($decoded)) { if ($decoded->Exception != '') { $xml_errors . '' . $decoded->Exception; }; }
              throw new \Exception('Unable to load response into object using simplexml.' . $xml_errors);

         }
        }

         //Convert json to object
         if ($contentType == 'json') {

          $decoded = json_decode($this->responseBody);
          if ($decoded->Exception != '') { throw new \Exception('addie\http\request\decode : ' . $decoded->Exception); };

         }

      
         return $decoded; //xml or json

      }


/*

    Get Functions

*/

    ## getStatusCode / Returns the http status code in the response.
    public function getStatusCode() {
      if ($this->httpCode == '') { throw new \Exception('No httpCode stored.'); };
      return $this->httpCode;
    }

    ## getFileSize / Returns the file size of the remote file.
    public function getFileSize() {
      if ($this->fileSize == '') { throw new \Exception('No fileSize stored.'); };
      return $this->fileSize;
    }

    ## getMimeType / Returns the mime type of the remote file.
    public function getMimeType() {
      if ($this->mimeType == '') { throw new \Exception('No mimeType stored.'); };
      return $this->mimeType;
    }

    ## getError / Returns the http error if any.
    public function getError() {
      return $this->error;
    }

    ## getHeader / Returns the header of the http response (raw).
    public function getHeader() {
      return $this->responseHeader;
    }

    ## getParsedHeader / Returns the header in an array format.
    public function parseHeader() {
      $r = $this->responseHeader;
      $o = array();
      $r = substr($r, strpos($r, "\r\n"));
      $r = explode("\r\n", $r);
      foreach ($r as $h) {
          list($v, $val) = explode(": ", $h);
          if ($v == null) continue;
          $o[$v] = $val;
      }
      return $o;
    }

    ## getBody / Returns the body of the http response.
    public function getBody() {
      return $this->responseBody;
    }

    ## getRaw / Returns the raw http response.
    public function getRaw() {
      return $this->raw;
    }
/*

    Private Functions

*/
    ## Check if url is valid
    private function check_url() {

        if (!isset($this->url)) { throw new \Exception("addie\http : url must be provided."); }

        $pos = strrpos($this->url, '/');
        $new_url = substr($this->url, 0, $pos);

        if (filter_var($new_url, FILTER_VALIDATE_URL) === FALSE) {
          throw new \Exception('addie\http : url is invalid');
        }
        
    }

    private function build_query_string() {

        $query_string = http_build_query($this->params, '', '&');
        return $query_string;

    }


    /*
     *
     * http_request - Makes the cURL request and stores the responce in a set of variables in the class object.
     */


  private function http_request($method) {

    // Set a default latency value.
    $latency = 0;

    //curl_init
    if ($method == 'GET') {
        if (!empty($this->params)) { $query_string = $this->build_query_string(); $ch = curl_init($this->url . '?' . $query_string);  };
        if (empty($this->params)) { $ch = curl_init($this->url); }; 
    }else{
        $ch = curl_init($this->url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "$method");
        if (isset($this->params) && is_array($this->params)) { curl_setopt($ch, CURLOPT_POSTFIELDS, $this->build_query_string()); }; //Send parameters
        if (isset($this->rawData)) { curl_setopt($ch, CURLOPT_POSTFIELDS, $this->rawData); }; //Send raw data
    }


    // Basic http authentication
    if (isset($this->credentials)) {
      curl_setopt($ch, CURLOPT_USERPWD, $this->credentials);
    }

    // Cookies
    if ($this->cookiePath != '') {
      curl_setopt($ch, CURLOPT_COOKIEJAR, $this->cookiePath);
      curl_setopt($ch, CURLOPT_COOKIEFILE, $this->cookiePath);
    }

    // Set header if provided.
    if (isset($this->headers)) {
      curl_setopt($ch, CURLOPT_HTTPHEADER, $this->headers);
    }

    //setSsl
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, $this->ssl);

    //timeout (these get added together)
    if (!empty($this->timeout)) {
      curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, '15');
      curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
    }

    //maxRedirects (Default 2)
    curl_setopt($ch, CURLOPT_MAXREDIRS, $this->max_redirects);

    // User agent (provided or default pre-defined)
    curl_setopt($ch, CURLOPT_USERAGENT, $this->userAgent);

    // Proxy
    if (isset($this->proxyUrl)) {
      curl_setopt($ch, CURLOPT_PROXY, $this->proxyUrl);
      if ($this->proxyUsername != '' && $this->proxyPassword != '') { $proxyAuth = $this->proxyUsername . ':' . $this->proxyPassword; curl_setopt($ch, CURLOPT_PROXYUSERPWD, $proxyAuth); };
    }

    //debug (Docs: https://curl.se/libcurl/c/CURLOPT_VERBOSE.html)
    curl_setopt($ch, CURLOPT_VERBOSE, $this->debug);

    //Static settings
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE); //See (https://curl.se/libcurl/c/CURLOPT_FOLLOWLOCATION.html) for hos this relates to CURLOPT_MAXREDIRS
    curl_setopt($ch, CURLOPT_HEADER, TRUE);

    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
    $file_size = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
    $mime_type = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
    $time = curl_getinfo($ch, CURLINFO_TOTAL_TIME);
    curl_close($ch);

    // Set the header, response, error and http code.
    $this->raw = $response;
    $this->responseHeader = substr($response, 0, $header_size);
    $this->responseBody = substr($response, $header_size);
    $this->error = $error;
    $this->httpCode = $http_code;
    $this->mimeType = $mime_type;
    $this->fileSize = $file_size;
    // Convert the latency to ms.
    $this->latency = round($time * 1000);

    if ($http_code == '201') { return true; }else{ return false; }  //Return true on false, either way responce is stored.

  }



} //End Class


/*
TESTING BELOW
*/
/*
$http = new http();
$http->setParams(array('name'=>'Don','last_name'=>'Hawkins'));
echo $http->request('https://yahoo.com','get');


$activate = new http();
$activate->setParams(array('name'=>'Don','last_name'=>'Hawkins'));
$activate->setCreds('username','password');
$res = $activate->request('POST');

//Changes just to commit
*/
?>
