<?php
namespace addie\auth;
use addie\http\request;

 /* 
            _     _ _   __     _             _               _ _   _     
           | |   | (_)  \ \   | |           (_)             (_) | | |    
   __ _  __| | __| |_  __\ \  | | ___   __ _ _ _ ____      ___| |_| |__  
  / _` |/ _` |/ _` | |/ _ \ \ | |/ _ \ / _` | | '_ \ \ /\ / / | __| '_ \ 
 | (_| | (_| | (_| | |  __/\ \| | (_) | (_| | | | | \ V  V /| | |_| | | |
  \__,_|\__,_|\__,_|_|\___| \_\_|\___/ \__, |_|_| |_|\_/\_/ |_|\__|_| |_|
                                        __/ |                            
                                       |___/                             

 *  What? A class that facilitates authenticating with external oauth services including Google, ebay, and others.
 *  Who? Created by Don Hawkins
 *  When? Started 5/15/2023
 */
 
        ### Sql Class (extends database)
        class LoginWith {

            public $client_id;
            public $client_secret;
            public $provider;

            public function __construct($client_id,$client_secret,$provider) {

                if ($client_id == '') { throw new \Exception("'client_id' is required."); }
                if ($client_secret == '') { throw new \Exception("The 'client_secret' string is required."); }

                $this->client_id = $client_id;
                $this->client_secret = $client_secret;
                $this->provider = $provider;
            }



        /*

        Create a the URL for redirection to the oauth provider.
            get_login_link($redirect_uri, $state, $scope))

        */
        public function get_login_link($redirect_uri,$state,$scope,$code_challenge) {

                //Google
                if (strtolower($this->provider) == 'google') {
                    $url = 'https://accounts.google.com/o/oauth2/v2/auth?';
                    $url .= 'client_id='.$this->client_id;
                    $url .= '&redirect_uri='.$redirect_uri;
                    $url .= '&response_type=code';
                    $url .= '&scope='. $this->convert_scope_to_url($scope);  //https://www.googleapis.com/auth/userinfo.email https://www.googleapis.com/auth/userinfo.profile
                    $url .= '&state='. 'google_' . $state;
                    return $url;
                }

                //Facebook
                if (strtolower($this->provider) == 'facebook') {    
                    $url = 'https://www.facebook.com/v19.0/dialog/oauth?';
                    $url .= 'client_id='.$this->client_id;
                    $url .= '&redirect_uri='.$redirect_uri;
                    $url .= '&response_type=code';
                    $url .= '&scope='. $this->convert_scope_to_url($scope); //email, public_profile (more info https://developers.facebook.com/docs/permissions)
                    $url .= '&state='. 'facebook_' . $state;
                    return $url;     
                    
                    /* ERROR Response:
                    YOUR_REDIRECT_URI?
                    error_reason=user_denied
                    &error=access_denied
                    &error_description=Permissions+error.
                    */
                }

                //Amazon
                if (strtolower($this->provider) == 'amazon') {
                    $url = 'https://www.amazon.com/ap/oa?';
                    $url .= 'client_id='.$this->client_id;
                    $url .= '&redirect_uri='.$redirect_uri;
                    $url .= '&response_type=code';
                    $url .= '&scope='. $this->convert_scope_to_url($scope);  //profile
                    $url .= '&state='. 'amazon_' . $state;
                    $url .= '&code_challenge='.$code_challenge;
                    return $url;
                }

        }



        /*

        Exchange the authorization 'code' for an access token.
             get_access_token($code,$redirect_uri) 
             Note: This function is called AFTER a code is returned but it still requires us to specify the $redirect_uri (same as passed originally)
            
                code_verifier is for Amazon only and is the same value as the 'code_challenge' passed to get_login_link()

        */

        public function get_access_token($code,$state,$redirect_uri,$code_verifier) {


            //Google
            if (str_contains($state,'google_') == 'google') {
                try { 
                    $http = new request('https://oauth2.googleapis.com/token'); 
                    $http->setParams(array('client_id' => $this->client_id,'client_secret' => $this->client_secret,'code' => $code,'grant_type' => 'authorization_code','redirect_uri' => $redirect_uri));
                    $http->setHeader(array('Content-Type' => 'application/x-www-form-urlencoded'));
                    $http->request('POST');
                    $body = $http->getBody('POST');
                    $decoded_json = $http->decode('json');
                    if ($decoded_json->error) { throw new \Exception('Error: ' . $decoded_json->error); }else{ $this->full_res = $decoded_json; $this->state = $state; $this->access_token = $decoded_json->access_token;  }
                    //print_r($http->getRaw());
                    return $decoded_json;
                } catch (Exception $e) { throw new \Exception('http error: ' . $e->getMessage()); }
            }

            //Facebook
            if (str_contains($state,'facebook_') == 'facebook') {
                try { 
                    $http = new request('https://graph.facebook.com/v19.0/oauth/access_token'); 
                    $http->setParams(array('client_id' => $this->client_id,'client_secret' => $this->client_secret,'code' => $code,'redirect_uri' => $redirect_uri));
                    $http->setHeader(array('Content-Type' => 'application/x-www-form-urlencoded'));
                    $http->request('GET');
                    $body = $http->getBody('POST');
                    $decoded_json = $http->decode('json');
                    if ($decoded_json->error->message) { throw new \Exception('Error: ' . strval($decoded_json->error->message)); }else{ $this->full_res = $decoded_json; $this->state = $state; $this->access_token = $decoded_json->access_token;  }
                    //print_r($http->getRaw());
                    return $decoded_json;
                } catch (Exception $e) { throw new \Exception('http error: ' . $e->getMessage()); }

                /* Json Returned
                {
                "access_token": {access-token}, 
                "token_type": {type},
                "expires_in":  {seconds-til-expiration}
                }
                */
            }

            //Amazon
            if (str_contains($state,'amazon_') == 'amazon') {
                try { 
                    $http = new request('https://api.amazon.com/auth/o2/token'); 
                    $http->setParams(array('client_id' => $this->client_id,'client_secret' => $this->client_secret,'code' => $code,'grant_type' => 'authorization_code','redirect_uri' => $redirect_uri,'code_verifier' => $code_verifier));
                    $http->setHeader(array('Content-Type' => 'application/x-www-form-urlencoded'));
                    $http->request('POST');
                    $body = $http->getBody('POST');
                    $decoded_json = $http->decode('json');
                    if ($decoded_json->error) { throw new \Exception('Error: ' . $decoded_json->error); }else{ $this->full_res = $decoded_json; $this->state = $state; $this->access_token = $decoded_json->access_token;  }
                    //print_r($http->getRaw());
                    return $decoded_json;
                } catch (Exception $e) { throw new \Exception('http error: ' . $e->getMessage()); }

                /*
        
                {
                    "access_token":"Atza|IQEBLjAsAhRmHjNgHpi0U-Dme37rR6CuUpSR...",
                    "token_type":"bearer",
                    "expires_in":3600,
                    "refresh_token":"Atzr|IQEBLzAtAhRPpMJxdwVz2Nn6f2y-tpJX2DeX..."
                }

                Error Res:
                ?error=access_denied&error_description=The+user+denied+your+request.&error_uri=https%3A%2F%2Fdeveloper.amazon.com%2Fdocs%2Famazon-login-identity-center%2Fauthenticate-users-with-amazon-
                */
            }
        }


        /*

        Make a request to a API, this is a private function that is called by the get_access_token function.
             get_user_info($full_res,$state,$url [only pass to get other information besides the basic userinfo]) 

        */
        public function get_user_info($url) {

            if (empty($this->state) || empty($this->full_res)) { throw new \Exception("First call 'get_login_link', then 'get_access_token' before calling the 'get_user_info' function."); }; 


            //Google
            if (str_contains($this->state,'google_')) {

                //Urls for google if you want to get stuff, if no url is provided the regular info is used by default

                    /*
                        People API, get contacts, phone numbers and morehttps://people.googleapis.com/v1/people/101348464557705166203?personFields=emailAddresses,names,addresses,phoneNumbers&access_token=ya29.a0AfB_byBNj_e8aF1Gm07BFFPxGv8aPkJjY5g2LIwubrD8WT8UwU5jeRSt7HTQus-uvHXKifLgih2oJf8-muiuN8YcDhzc7BmXbMJDrJfXEbkXVNdkzKQ2zl5roZM0rlqASv0Z-OZjbZlurXs8yhpk6WYHkCSc7T9n0mFXaCgYKAWgSARASFQHGX2Miowq6886nVOCbJFXXoqhcOA0171
                        https://developers.google.com/people/api/rest/v1/people/get

                        Regular Info (email, name, etc)
                        https://www.googleapis.com/oauth2/v2/userinfo?access_token=XX

                    */

                    if ($url == '') { $url = 'https://www.googleapis.com/oauth2/v2/userinfo?access_token=' . urlencode($this->access_token); }else{ $url = $url; };
                    $http = new request($url);
                    $http->request('GET'); 
                    $body = $http->getBody('GET');
                    $decoded_json = $http->decode('json');

                    $return_array = array(
                        "access_token" => $this->access_token,                      
                        "first_name" => $decoded_json->given_name,
                        "last_name" => $decoded_json->family_name,
                        "email" => $decoded_json->email,
                        "profile_pic" => $decoded_json->picture,
                        "id" => $decoded_json->id,
                        "full_res" => $decoded_json
                    );

                    return $return_array;

            }
            
            //Facebook
            if (str_contains($this->state,'facebook_')) {

                //Urls for google if you want to get stuff, if no url is provided the regular info is used by default

                    /*
                        https://graph.facebook.com/me?access_token=<your-fb-token>&
                        fields=id,name,email,picture.width(640).height(640)

                    */

                    if ($url == '') { $url = 'https://graph.facebook.com/me?access_token=' . urlencode($this->access_token) . '&fields=id,name,email,picture.width(640).height(640)'; }else{ $url = $url; };
                    $http = new request($url);
                    $http->request('GET'); 
                    $body = $http->getBody('GET');
                    $decoded_json = $http->decode('json');

                    $return_array = array(
                        "access_token" => $this->access_token,                      
                        "first_name" => explode(' ',$decoded_json->name)[0],
                        "last_name" => explode(' ',$decoded_json->name)[1],
                        "email" => $decoded_json->email,
                        "profile_pic" => $decoded_json->picture->data->url,
                        "id" => $decoded_json->id,
                        "full_res" => $decoded_json
                    );

                    return $return_array;

            }


            //Amazon
            if (str_contains($this->state,'amazon_')) {


                    if ($url == '') { $url = 'https://api.amazon.com/user/profile?access_token=' . urlencode($this->access_token); }else{ $url = $url; };
                    $http = new request($url);
                    $http->request('GET'); 
                    $body = $http->getBody('GET');
                    $decoded_json = $http->decode('json');
                    if ($decoded_json->error != '') { throw new Exception($decoded_json->error . ' : ' . $decoded_json->error_description); } 

                    $return_array = array(
                        "access_token" => $this->access_token,                      
                        "first_name" => explode(' ',$decoded_json->name)[0],
                        "last_name" => explode(' ',$decoded_json->name)[1],
                        "email" => $decoded_json->email,
                        "postal_code" => $decoded_json->postal_code,
                        "full_res" => $decoded_json
                    );

                    return $return_array;

            }


        }

        /*
        Private Functions
        */
        private function convert_scope_to_url(array $array) {
            foreach ($array as $key => $value) {
                $url .= $value.'+';
            }
            return $url;
        }

    } //End Class






//https://www.googleapis.com/oauth2/v1/tokeninfo?access_token=XXX

//https://people.googleapis.com/v1/people/101348464557705166203?personFields=addresses,phoneNumbers&access_token=

